"""Business logic for managing employees."""

from __future__ import annotations

from typing import List, Optional, Any, Dict
from datetime import date

# Local imports (no package prefix)
from repositories.employee_repo import EmployeeRepository
from models.employee import Employee


class EmployeeService:
    def __init__(self, repo: Optional[EmployeeRepository] = None) -> None:
        self.repo = repo or EmployeeRepository()

    # -------- internal helpers --------
    def _list_all(self) -> List[Dict[str, Any]]:
        """Be resilient to different repo method names."""
        if hasattr(self.repo, "list_all"):
            return self.repo.list_all()
        if hasattr(self.repo, "get_all"):
            return self.repo.get_all()  # type: ignore[no-any-return]
        if hasattr(self.repo, "all"):
            return self.repo.all()  # type: ignore[no-any-return]
        if hasattr(self.repo, "table"):  # TinyDB fallback
            return [dict(r) for r in self.repo.table.all()]
        return []

    # -------- public API --------
    def list_employees(self, active_only: bool = False) -> List[Dict[str, Any]]:
        if active_only:
            if hasattr(self.repo, "active_employees"):
                return self.repo.active_employees()  # type: ignore[no-any-return]
            # Fallback: filter manually
            rows = self._list_all()
            return [r for r in rows if r.get("is_active") in (True, None)]
        return self._list_all()

    def add_employee(self, employee: Employee | Dict[str, Any]) -> int:
        """Create employee; accepts Pydantic model or dict."""
        if hasattr(employee, "dict"):
            return self.repo.create_employee(employee)  # type: ignore[arg-type]
        return self.repo.create_employee(employee)  # type: ignore[arg-type]

    def update_employee(self, emp_id: int, **fields: Any) -> bool:
        return self.repo.update_employee(emp_id, **fields)

    def delete_employee(self, emp_id: int) -> bool:
        return self.repo.delete_employee(emp_id)

    def get_employee(self, emp_id: int) -> Optional[Dict[str, Any]]:
        return self.repo.get_by_id(emp_id)

    def find_by_emp_code(self, emp_code: str) -> Optional[Dict[str, Any]]:
        if hasattr(self.repo, "find_by_emp_code"):
            return self.repo.find_by_emp_code(emp_code)  # type: ignore[no-any-return]
        # Fallback search
        for r in self._list_all():
            if r.get("emp_code") == emp_code:
                return r
        return None
